#!/bin/bash
if [ "$EUID" -ne 0 ]; then
  echo "Por favor, execute como root"
  exit 1
fi

limpar_interfaces() {
  echo "Limpando todas as interfaces de rede..."

  rm -f /etc/netplan/*

  ip addr flush scope global

  echo "Todas as interfaces foram limpas."
}

criar_configuracao_netplan() {
  echo "Criando configuração netplan..."
  mkdir -p /etc/netplan

  rm -f /etc/netplan/*

  NETPLAN_CONFIG="/etc/netplan/01-network-config.yaml"

  echo "network:" > $NETPLAN_CONFIG
  echo "  version: 2" >> $NETPLAN_CONFIG
  echo "  renderer: networkd" >> $NETPLAN_CONFIG
  echo "  ethernets:" >> $NETPLAN_CONFIG
  echo "    eth0:" >> $NETPLAN_CONFIG

  if [ -z "$IP_ADDRESS" ]; then
    echo "      dhcp4: true" >> $NETPLAN_CONFIG
    echo "      dhcp6: false" >> $NETPLAN_CONFIG
    echo "      nameservers:" >> $NETPLAN_CONFIG
    echo "        addresses: [$NAMESERVERS]" >> $NETPLAN_CONFIG
  else
    echo "      dhcp4: false" >> $NETPLAN_CONFIG
    echo "      dhcp6: false" >> $NETPLAN_CONFIG
    echo "      addresses: [$IP_ADDRESS/$NETMASK]" >> $NETPLAN_CONFIG
    echo "      routes:" >> $NETPLAN_CONFIG
    echo "        - to: default" >> $NETPLAN_CONFIG
    echo "          via: $GATEWAY" >> $NETPLAN_CONFIG
    echo "      nameservers:" >> $NETPLAN_CONFIG
    echo "        addresses: [$NAMESERVERS]" >> $NETPLAN_CONFIG
  fi

  chmod 600 "$NETPLAN_CONFIG"

  echo "Configuração netplan criada."
}

aplicar_configuracao_netplan() {
  echo "Aplicando configuração netplan..."
  netplan apply
  sleep 3
  echo "Configuração netplan aplicada."
}

atualizar_hostname() {
  echo "Atualizando hostname para $NOVO_HOSTNAME..."
  echo "$NOVO_HOSTNAME" > /etc/hostname
  if grep -q "127.0.1.1" /etc/hosts; then
    sed -i "s/127.0.1.1 .*/127.0.1.1 $NOVO_HOSTNAME/g" /etc/hosts
  else
    echo "127.0.1.1 $NOVO_HOSTNAME" >> /etc/hosts
  fi
  hostnamectl set-hostname "$NOVO_HOSTNAME"
  hostname "$NOVO_HOSTNAME"
  current_hostname=$(hostname)
  if [ "$current_hostname" != "$NOVO_HOSTNAME" ]; then
    echo "AVISO: Hostname não foi configurado corretamente. Tentando método alternativo..."
    hostname "$NOVO_HOSTNAME"
  fi
  echo "Hostname atualizado para: $(hostname)"
}

testar_conectividade() {
  echo "Testando conectividade..."
  if ping -c 3 8.8.8.8 > /dev/null 2>&1; then
    DNS_STATUS="Sucesso"
  else
    DNS_STATUS="Falha"
  fi
  if curl -s --head --connect-timeout 10 http://condorflix-dev.web.app/ > /dev/null 2>&1; then
    WEB_STATUS="Sucesso"
  else
    WEB_STATUS="Falha"
  fi
  echo "Testes completos."
}

install_mesh_and_packages() {
  echo "Instalando pacotes necessários..."
  SCRIPT_DIR="$(dirname "$(readlink -f "$0")")"
  PKG_DIR="${SCRIPT_DIR}/pkgs"

  mkdir -p /tmp/packages
  cd /tmp/packages

  if [ -f "${PKG_DIR}/net-tools_2.10-0.1ubuntu4_amd64.deb" ]; then
    echo "Usando net-tools do diretório local..."
    cp "${PKG_DIR}/net-tools_2.10-0.1ubuntu4_amd64.deb" .
   fi

  if [ -f "${PKG_DIR}/traceroute_1%3a2.1.5-1_amd64.deb" ]; then
    echo "Usando traceroute do diretório local..."
    cp "${PKG_DIR}/traceroute_1%3a2.1.5-1_amd64.deb" .
   fi


  echo "Instalando pacotes no sistema atual..."
  dpkg -i net-tools_2.10-0.1ubuntu4_amd64.deb
  dpkg -i traceroute_1%3a2.1.5-1_amd64.deb
  apt-get -f install -y

  echo "Instalando MeshAgent..."
  chmod 755 /root/meshinstall.sh && sudo -E /root/meshinstall.sh https://mesh.megamidia.com.br 'GCmFaqIid80PhCYnqfwKqEdtVaCOrG5YvNY@noom62EWptluYitMJx1jK@Yk1Ne0' || /root/meshinstall.sh https://mesh.megamidia.com.br 'GCmFaqIid80PhCYnqfwKqEdtVaCOrG5YvNY@noom62EWptluYitMJx1jK@Yk1Ne0'

  echo "Montando sistema persistente para instalação de pacotes..."
  mount -o remount,rw /media/root-ro

  cp "${SCRIPT_DIR}/fix.sh" "/media/root-ro/root/"

  echo "Instalando pacotes no sistema persistente via chroot..."
  mkdir -p /media/root-ro/mnt/packages
  cp -a /tmp/packages/*.deb /media/root-ro/mnt/packages/

  mkdir -p /media/root-ro/proc
  mkdir -p /media/root-ro/sys
  mkdir -p /media/root-ro/dev
  mkdir -p /media/root-ro/dev/pts

  mount -t proc proc /media/root-ro/proc
  mount -t sysfs sys /media/root-ro/sys
  mount -o bind /dev /media/root-ro/dev
  mount -o bind /dev/pts /media/root-ro/dev/pts

  cp /etc/resolv.conf /media/root-ro/etc/resolv.conf

  echo "Executando instalação dentro do chroot..."
  chroot /media/root-ro bash -c "dpkg -i /mnt/packages/net-tools_2.10-0.1ubuntu4_amd64.deb || true"
  chroot /media/root-ro bash -c "dpkg -i /mnt/packages/traceroute_1%3a2.1.5-1_amd64.deb || true"
  chroot /media/root-ro bash -c "apt-get -f install -y"

  echo "Limpando ambiente..."
  umount /media/root-ro/proc
  umount /media/root-ro/sys
  umount /media/root-ro/dev/pts
  umount /media/root-ro/dev
  rm -rf /media/root-ro/mnt/packages

  echo "MeshAgent e pacotes instalados com sucesso no sistema persistente."
}

persistir_configuracao_netplan() {
  echo "Limpando e persistindo configurações..."
  if ! mount | grep -q "/media/root-ro" || mount | grep -q "/media/root-ro.*ro"; then
    mount -o remount,rw /media/root-ro
  fi

  rm -rf /media/root-ro/etc/netplan/* /media/root-ro/etc/netplan/.[!.]* 2>/dev/null || true

  mkdir -p /media/root-ro/etc/netplan
  rsync -a /etc/netplan/ /media/root-ro/etc/netplan/
  rsync -a /etc/hostname /media/root-ro/etc/hostname
  rsync -a /etc/hosts /media/root-ro/etc/hosts

  if [ -d "/usr/local/mesh_services" ]; then
    mkdir -p /media/root-ro/usr/local/mesh_services/
    rm -rf /media/root-ro/usr/local/mesh_services/* /media/root-ro/usr/local/mesh_services/.[!.]* 2>/dev/null || true
    rsync -a /usr/local/mesh_services/ /media/root-ro/usr/local/mesh_services/
  fi

  if [ -f "/usr/lib/systemd/system/meshagent.service" ]; then
    mkdir -p /media/root-ro/usr/lib/systemd/system/
    rsync -a /usr/lib/systemd/system/meshagent.service /media/root-ro/usr/lib/systemd/system/
  fi

  if [ -f "/etc/systemd/system/multi-user.target.wants/meshagent.service" ]; then
    mkdir -p /media/root-ro/etc/systemd/system/multi-user.target.wants/
    rsync -a /etc/systemd/system/multi-user.target.wants/meshagent.service /media/root-ro/etc/systemd/system/multi-user.target.wants/
  fi

  # echo "Remontando sistema persistente como somente leitura..."
  # timeout 15s mount -o remount,ro /media/root-ro

  echo "Configurações persistidas com sucesso."
}

exec 3>&1
INPUT=$(dialog --ok-label "Enviar" \
  --backtitle "Configuração de Rede" \
  --title "Configuração de Rede" \
  --form "Por favor, insira suas configurações de rede para eth0:" 15 70 0 \
  "Endereço IP (deixe em branco para DHCP):" 1 1 "" 1 43 20 0 \
  "Máscara de Sub-rede (padrão: 24):" 2 1 "24" 2 43 20 0 \
  "Gateway (padrão: 192.168.1.1):" 3 1 "192.168.1.1" 3 43 20 0 \
  "Servidor de DNS (padrão: 8.8.8.8):" 4 1 "8.8.8.8" 4 43 20 0 \
  2>&1 1>&3)
dialog_exit_status=$?
exec 3>&-

if [ $dialog_exit_status -ne 0 ]; then
  dialog --msgbox "Operação cancelada. Nenhuma mudança será realizada." 6 40
  exit 1
fi

IP_ADDRESS=$(echo "$INPUT" | sed -n 1p)
NETMASK=$(echo "$INPUT" | sed -n 2p)
GATEWAY=$(echo "$INPUT" | sed -n 3p)
NAMESERVERS=$(echo "$INPUT" | sed -n 4p)
[ -z "$NAMESERVERS" ] && NAMESERVERS="8.8.8.8"

NOVO_HOSTNAME=$(dialog --inputbox "Por favor, insira o novo nome do host:" 8 40 2>&1 >/dev/tty)
dialog_exit_status=$?

if [ $dialog_exit_status -ne 0 ]; then
  dialog --msgbox "Operação cancelada. Nenhuma mudança será realizada." 6 40
  exit 1
fi

dialog --yesno "As seguintes configurações serão aplicadas:\n\n\
Endereço IP: $IP_ADDRESS\n\
Máscara de Sub-rede: $NETMASK\n\
Gateway: $GATEWAY\n\
Servidor de DNS: $NAMESERVERS\n\
Nome do Host: $NOVO_HOSTNAME\n\n\
Deseja continuar com a configuração?" 15 60

if [ $? -ne 0 ]; then
  dialog --msgbox "Operação cancelada. Nenhuma mudança será realizada." 6 40
  exit 1
fi

clear

limpar_interfaces
criar_configuracao_netplan
atualizar_hostname
aplicar_configuracao_netplan
testar_conectividade

dialog --title "Resultados de Conectividade" \
  --colors \
  --msgbox "\n\
Teste de conectividade DNS (8.8.8.8): \Z1$DNS_STATUS\Zn\n\
Teste de conectividade Web (condorflix-dev): \Z1$WEB_STATUS\Zn\n\n\
Hostname configurado: $NOVO_HOSTNAME\n\
IP configurado: $([ -z "$IP_ADDRESS" ] && echo "DHCP" || echo "$IP_ADDRESS/$NETMASK")\n\
DNS configurado: $NAMESERVERS" 12 60

dialog --yesno "Deseja salvar todas as configurações e reiniciar o sistema?\n\n\
Selecione 'Sim' para salvar e reiniciar, ou 'Não' para cancelar." 10 60

if [ $? -eq 0 ]; then
  install_mesh_and_packages
  persistir_configuracao_netplan
  dialog --msgbox "Configurações e pacotes salvos com sucesso. O sistema será reiniciado." 6 60
  reboot
else
  dialog --msgbox "Configurações não foram salvas. As mudanças não serão persistentes após o reboot." 6 40
fi

exit 0
